import acm.program.*;
import acm.util.*;
import java.io.*;
import java.util.*;

public class WordSkip extends ConsoleProgram {
	/* The name of the dictionary file. */
	private static final String WORDS_FILE = "dictionary.txt";
	
	public void run() {
		/* Let's see just how cool a skip this is! */
		setFont("DejaVuSerif-BOLD-24");
		
		/* Build up a map from letter sequences to the words that start with those
		 * sequences.
		 */
		HashMap<Character, ArrayList<String>> dictionary = loadDictionary();
		
		/* Continuously ask the user for a word, then do a word walk! */
		while (true) {
			String word = readLine("Enter starting word: ").toUpperCase();
			doWordSkip(word, dictionary);
		}
	}
	
	/**
	 * Given a word and a dictionary, does a word skip from the starting word through
	 * the dictionary.
	 * 
	 * @param word The word to start from.
	 * @param dictionary The dictionary of words.
	 */
	private void doWordSkip(String word, HashMap<Character, ArrayList<String>> dictionary) {
		RandomGenerator rgen = RandomGenerator.getInstance();
		HashSet<String> used = new HashSet<String>();
		
		while (true) {
			/* Print the current word. */
			println(word);
			
			/* Look up what the successor words are. */
			char suffix = word.charAt(word.length() - 1);
			ArrayList<String> options = dictionary.get(suffix);
			
			/* If no words start with this sequence, we're done. */
			if (options == null)
				break;
			
			/* Otherwise, choose a random successor word. */
			word = options.get(rgen.nextInt(0, options.size() - 1));
		
			/* If we've used this word, we're done. */
			if (used.contains(word)) break;
		
			/* Mark that we've now used it. */
			used.add(word);
		}
		
		println("This walk took: " + used.size() + " words.");
	}
	
	/**
	 * Reads in the dictionary as a map from prefixes to words with that prefix.
	 * 
	 * @return The dictionary, sorted by prefixes.
	 */
	private HashMap<Character, ArrayList<String>> loadDictionary() {
		try {
			BufferedReader br = new BufferedReader(new FileReader(WORDS_FILE));
			HashMap<Character, ArrayList<String>> result = new HashMap<Character, ArrayList<String>>();
			
			while (true) {
				String line = br.readLine();
				if (line == null) break;
				
				/* Determine the prefix of the current word. */
				char prefix = line.charAt(0);
				
				/* Ensure that there is something in the HashMap paired with this prefix. */
				if (!result.containsKey(prefix))
					result.put(prefix, new ArrayList<String>());
				
				/* Add the current word to the list of strings beginning with its prefix. */
				result.get(prefix).add(line);
			}
			
			br.close();
			return result;			
		} catch (IOException e) {
			throw new ErrorException(e);
		}
	}
}
